/*____________________________________________________________________________
	Copyright (C) 2001 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: PGPtclIB.c,v 1.3.2.2 2001/04/24 21:43:34 ajivsov Exp $
____________________________________________________________________________*/
/* PGP TCL module for iButton (custom applet version) */
/* Assumes PGPkey applet is selected and running */

#include "pgpSDKBuildFlags.h"

#include <windows.h>

#include "pgpConfig.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "pgpTokenLib.h"
#include "jibapi.h"

#define PGPKEY_CLA				(BYTE)0xB5
#define PGPKEY_INS_INIT			(BYTE)0
#define PGPKEY_INS_SETPIN		(BYTE)1
#define PGPKEY_INS_SETMODULUS	(BYTE)2
#define PGPKEY_INS_GETMODULUS	(BYTE)3
#define PGPKEY_INS_SETEXPONENT	(BYTE)4
#define PGPKEY_INS_SETKEYID		(BYTE)5
#define PGPKEY_INS_GETKEYID		(BYTE)6
#define PGPKEY_INS_EXPONENTIATE	(BYTE)7
#define PGPKEY_INS_SETBLOB		(BYTE)8
#define PGPKEY_INS_APPENDBLOB	(BYTE)9
#define PGPKEY_INS_GETBLOB		(BYTE)10
#define PGPKEY_INS_RSAKEYGEN	(BYTE)11

#define NKEYS	2

#define LOAD_UNLOAD_TIME  500
#define EXECUTE_TIME      5000
#define KEYGEN_TIME		  0xffff

#define KEYIDSIZE			8
#define MAXCMDSIZE			0xf0

#define MIN(x,y) (((x)<=(y))?(x):(y))


/* serial ports to search for */
#ifdef WIN32
char *g_DeviceNames[] = { "COM1", "COM2", "COM3", "COM4"};
BYTE g_SerialPortCount = 4;
#else
char *g_DeviceNames[] = { "/dev/ttya" };
BYTE g_SerialPortCount = 1;
#endif

extern struct PGPToken ourToken;

static PGPBoolean gottoken;
static int nkeys = 0;
static PGPTokenKeyInfo keyinfo[NKEYS];
static PGPByte tokenKeyids[NKEYS][KEYIDSIZE];
static PGPTokenKeyInfo pubkeyinfo;
static PGPByte *tokenKeyidBlob = &pubkeyinfo.keyid[0];
static int tokenPinChecked = 0;
static BYTE tokenPin[256];
static int tokenPinLen;
/* array of serial numbers */
static LPBYTE  roms;


/* Look for a matching keyid on the token */
static int
ibKeyIDIndex( PGPByte const *keyid )
{
	int i;

	for (i=0; i<nkeys; ++i)
	{
		if( memcmp( tokenKeyids[i], keyid, KEYIDSIZE ) == 0 )
			return i;
	}
	return -1;
}



/* PGPToken struct functions */


static int
ibInit( PGPToken *tptr, PGPUInt32 whichslot, PGPBoolean haveSlot )
{
	return 0;
}

static int
ibTerm( PGPToken *tptr )
{
	return 0;
}

static int
ibDecrypt( PGPToken *tptr, PGPByte *keyid, unsigned char *in, PGPSize isize,
		  unsigned char *out, PGPSize osize )
{
	LPRESPONSEAPDU  apdu = NULL;
	BYTE bigbuf[1000];
	unsigned buflen;
	unsigned i;

	if( isize + tokenPinLen + 2 > sizeof(bigbuf) )
		return -1;

	bigbuf[0] = tokenPinLen;
	memcpy( bigbuf+1, tokenPin, tokenPinLen );
	bigbuf[tokenPinLen+1] = (PGPByte)ibKeyIDIndex(keyid);
	memcpy( bigbuf+2+tokenPinLen, in, isize );

	apdu = Process(PGPKEY_CLA, PGPKEY_INS_EXPONENTIATE,
		0, 0, (BYTE)(tokenPinLen+2+isize), bigbuf, EXECUTE_TIME);

	if(apdu->SW != (WORD)0x9000)
		return -1;

	buflen = apdu->Len;
	if( buflen > isize )
		return -2;
	if( buflen < isize )
		memset( bigbuf, 0, isize-buflen );
	memcpy( bigbuf+isize-buflen, apdu->Data, buflen );
	
	/* Do PKCS-1 decoding */
	if (bigbuf[0] != 0 || bigbuf[1] != 2)
		return -1;
	for( i=3; i<isize; ++i )
		if (bigbuf[i] == 0)
			break;
	if (i == isize)
		return -1;

	i += 1;
	if (isize-i > osize)
		return -1;

	memcpy( out, bigbuf+i, isize-i );
	return isize-i;
}

static int
ibSign( PGPToken *tptr, PGPByte *keyid, unsigned char *in, PGPSize isize,
	   unsigned char *out, PGPSize osize )
{
	LPRESPONSEAPDU  apdu = NULL;
	BYTE bigbuf[1000];
	unsigned i;

	if( osize + tokenPinLen + 2 > sizeof(bigbuf) )
		return -1;

	bigbuf[0] = tokenPinLen;
	memcpy( bigbuf+1, tokenPin, tokenPinLen );
	bigbuf[tokenPinLen+1] = (PGPByte)ibKeyIDIndex(keyid);

	/* Set up PKCS padding */
	bigbuf[2+tokenPinLen] = 0;
	bigbuf[3+tokenPinLen] = 1;
	for( i=0; i<osize-isize-3; ++i)
		bigbuf[4+i+tokenPinLen] = 0xff;
	bigbuf[osize-isize+1+tokenPinLen] = 0;
	memcpy( bigbuf+osize-isize+2+tokenPinLen, in, isize );

	apdu = Process(PGPKEY_CLA, PGPKEY_INS_EXPONENTIATE,
		0, 0, (BYTE)(tokenPinLen+2+osize), bigbuf, EXECUTE_TIME);

	if(apdu->SW != (WORD)0x9000)
		return -1;

	if (apdu->Len > osize )
		return -1;
	if( apdu->Len < osize )
		memset( out, 0, osize - apdu->Len );
	memcpy( out + osize - apdu->Len,
				apdu->Data, apdu->Len );
	return 0;
}


/* Return 0 if phrase is correct, 1 if incorrect */
static int
ibAuth( PGPToken *tptr, char const *pin, PGPSize length )
{
	LPRESPONSEAPDU  apdu = NULL;
	BYTE bigbuf[1000];

	if( tokenPinChecked )
	{
		if( (int)length == tokenPinLen &&
			memcmp( pin, tokenPin, length ) == 0 )
			return 0;
		return 1;
	}

	bigbuf[0] = (BYTE) length;
	memcpy( bigbuf+1, pin, length );
	memcpy( bigbuf+1+length, pin, length );

	apdu = Process(PGPKEY_CLA, PGPKEY_INS_SETPIN,
			0, 0, (BYTE)(1+2*length), bigbuf, LOAD_UNLOAD_TIME);

	 if(apdu->SW != (WORD)0x9000)
		 return 1;
	 memcpy( tokenPin, pin, length );
	 tokenPinLen = length;
	 tokenPinChecked = 1;
	 return 0;
}

static int
ibDeauth( PGPToken *tptr )
{
	return 0;
}


static PGPTokenKeyInfo *
ibGetPrivKeyIDs(PGPToken *tok, PGPSize *n)
{
	PGPTokenKeyInfo *p = malloc( nkeys * sizeof(PGPTokenKeyInfo) );
	memcpy( p, keyinfo, nkeys * sizeof(PGPTokenKeyInfo) );
	*n = nkeys;
	return p;
}

static PGPTokenKeyInfo *
ibGetPubKeyIDs(PGPToken *tok, PGPSize *n)
{
	PGPTokenKeyInfo *p;
	PGPByte zeroid[KEYIDSIZE];
	memset( zeroid, 0, KEYIDSIZE );
	if( memcmp( &pubkeyinfo.keyid, zeroid, KEYIDSIZE ) == 0 )
	{
		*n = 0;
		return NULL;
	}
	*n = 1;
	p = malloc( sizeof(PGPTokenKeyInfo) );
	memcpy( p, &pubkeyinfo, sizeof(PGPTokenKeyInfo) );
	return p;
}

static int
ibPutPrivate(PGPToken *tptr, PGPBoolean isMaster,
					 unsigned char *ID,
					 unsigned char *modulus,
					 PGPSize modulus_size,
					 unsigned char *public_exponent,
					 PGPSize public_exponent_size,
					 unsigned char *private_exponent,
					 PGPSize private_exponent_size,
					 unsigned char *prime_1,
					 PGPSize prime_1_size,
					 unsigned char *prime_2,
					 PGPSize prime_2_size,
					 unsigned char *exponent_1,
					 PGPSize exponent_1_size,
					 unsigned char *exponent_2,
					 PGPSize exponent_2_size,
					 unsigned char *coefficient,
					 PGPSize coefficient_size)
{
  LPRESPONSEAPDU  apdu;
  BYTE			  cmdbuf[1000];
  int			  keyindex = 0;

  if (nkeys >= NKEYS)
	  return -1;

  keyindex = nkeys;

  /* Set PIN in cmdbuf for remaining functions */
  cmdbuf[0] = tokenPinLen;
  memcpy( cmdbuf+1, tokenPin, tokenPinLen );
  cmdbuf[tokenPinLen+1] = keyindex;

  /* Set modulus */
  memcpy(cmdbuf+tokenPinLen+2, modulus, modulus_size);
  apdu = Process(PGPKEY_CLA, PGPKEY_INS_SETMODULUS,
	  0, 0, (BYTE)(tokenPinLen+2+modulus_size), cmdbuf, LOAD_UNLOAD_TIME);

  if(apdu->SW != (WORD)0x9000)
  {
    return -2;
  }
  
  /* Set exponent */
  memcpy(cmdbuf+tokenPinLen+2, private_exponent, private_exponent_size);
  apdu = Process(PGPKEY_CLA, PGPKEY_INS_SETEXPONENT,
	  0, 0, (BYTE)(tokenPinLen+2+private_exponent_size), cmdbuf, LOAD_UNLOAD_TIME);

  if(apdu->SW != (WORD)0x9000)
  {
    return -3;
  }

  /* Set keyid */
  memcpy(cmdbuf+tokenPinLen+2, ID, 8);
  apdu = Process(PGPKEY_CLA, PGPKEY_INS_SETKEYID,
	  0, 0, (BYTE)(tokenPinLen+2+8), cmdbuf, LOAD_UNLOAD_TIME);

  if(apdu->SW != (WORD)0x9000)
  {
    printf("SetKeyID failed with SW = %04x\n",apdu->SW);
    return -3;
  }

  /* Put public exponent into blob */
	memcpy( cmdbuf, ID, KEYIDSIZE );
	cmdbuf[KEYIDSIZE+0] = 0x00;
	cmdbuf[KEYIDSIZE+1] = 0x03;
	cmdbuf[KEYIDSIZE+2] = 0x01;
	cmdbuf[KEYIDSIZE+3] = 0x00;
	cmdbuf[KEYIDSIZE+4] = 0x01;
	apdu = Process(PGPKEY_CLA, PGPKEY_INS_SETBLOB,
		0, 0, KEYIDSIZE+2+3, cmdbuf, LOAD_UNLOAD_TIME);
	if(apdu->SW != (WORD)0x9000)
	{
		printf("SetBlob failed with SW = %04x\n",apdu->SW);
		return -3;
	}

  memcpy( tokenKeyids[keyindex], ID, KEYIDSIZE );
  memcpy( keyinfo[keyindex].keyid, ID, KEYIDSIZE );
  memcpy( tokenKeyidBlob, ID, KEYIDSIZE );
  ++nkeys;
  return 0;

}

static int
ibPutPublic( PGPToken *tptr, BYTE *keyid, BYTE *data, PGPSize size,
			pgpTokenDataInfo *info )
{
  LPRESPONSEAPDU  apdu = NULL;
  BYTE			  cmdbuf[1000];
  BYTE			  *cmdptr;
  BYTE			  cmd;
  int			  explen;

  memcpy( cmdbuf, keyid, KEYIDSIZE );

  apdu = Process(PGPKEY_CLA, PGPKEY_INS_GETBLOB,
		0, 0, 0, NULL, LOAD_UNLOAD_TIME);
  if(apdu->SW != (WORD)0x9000)
  {
    printf("GetBlob failed with SW = %04x\n",apdu->SW);
    return -1;
  }

  explen = (apdu->Data[0]<<8) | apdu->Data[1];
  memcpy( cmdbuf + KEYIDSIZE, apdu->Data, explen+2 );
  memcpy( cmdbuf+KEYIDSIZE+explen+2, &info->pubKeyStub,
		  sizeof(pgpTokenPubKeyStub) );
  memcpy( cmdbuf+KEYIDSIZE+explen+2+sizeof(pgpTokenPubKeyStub),
			data, size );

  size += KEYIDSIZE+explen+2+sizeof(pgpTokenPubKeyStub);
  cmd = PGPKEY_INS_SETBLOB;
  cmdptr = cmdbuf;
  while( size > 0 )
  {
	apdu = Process(PGPKEY_CLA, cmd,
		0, 0, (BYTE)MIN(MAXCMDSIZE,size), cmdptr, LOAD_UNLOAD_TIME);
	if(apdu->SW != (WORD)0x9000)
	{
		printf("SetBlob failed with SW = %04x\n",apdu->SW);
		return -3;
	}
	cmdptr += MIN(MAXCMDSIZE, size);
	size -= MIN (MAXCMDSIZE, size);
	cmd = PGPKEY_INS_APPENDBLOB;
  }
  memcpy( tokenKeyidBlob, keyid, KEYIDSIZE );
  return 0;
}

/* The public blob consists of 2 bytes of public exponent
 * length in bytes; the public exponent; the
 * PGPTokenPubKeyStub; and the blob data.
 */
static BYTE *
ibGetPublic( PGPToken *tptr, BYTE *keyid, PGPSize *size,
			pgpTokenDataInfo **info )
{
  LPRESPONSEAPDU  apdu = NULL;
  PGPUInt32 modlen, explen;
  PGPByte *mod;
  pgpTokenDataInfo *datainfo;
  BYTE cmdbuf[1];

  *size = 0;
  *info = NULL;

  if (memcmp(tokenKeyidBlob, keyid, 8) != 0)
	  return NULL;

  /* Get Modulus */
   cmdbuf[0] = ibKeyIDIndex( keyid );
 apdu = Process(PGPKEY_CLA, PGPKEY_INS_GETMODULUS,
				 0, 0, 1, cmdbuf, LOAD_UNLOAD_TIME);
  if(apdu->SW != (WORD)0x9000)
	return NULL;

  modlen = apdu->Len;
  mod = malloc( modlen );
  memcpy( mod, apdu->Data, modlen );

  /* Get blob */
  apdu = Process(PGPKEY_CLA, PGPKEY_INS_GETBLOB,
				0, 0, 0, NULL, LOAD_UNLOAD_TIME);
  
  if(apdu->SW != (WORD)0x9000) {
	free( mod );
	return NULL;
  }
  
  explen = (apdu->Data[0]<<8) | apdu->Data[1];

  datainfo = malloc( sizeof(pgpTokenDataInfo) + modlen + explen );
  datainfo->alg = kPGPPublicKeyAlgorithm_RSA;
  datainfo->exp_size = explen;
  datainfo->mod_size = modlen;
  memcpy( datainfo + sizeof(pgpTokenDataInfo), apdu->Data+2, explen );
  memcpy( datainfo + sizeof(pgpTokenDataInfo) + explen, mod, modlen );
  free( mod );

  memcpy( &datainfo->pubKeyStub, apdu->Data+2+explen,
		  sizeof(pgpTokenPubKeyStub) );

  *info = datainfo;
  *size = apdu->Len - 2 - explen - sizeof(pgpTokenPubKeyStub);
  mod = malloc( *size );
  memcpy( mod, apdu->Data + 2 + explen + sizeof(pgpTokenPubKeyStub),
	  *size );
  return mod;
}

/* TODO: implement removal of all objects related to keyid. 
   if is_last=true, then it is a last call in a series. 

   For V4 keys we will have one call for the master key and then 
   one for (each) subkey.
  */
static int
ibDel( PGPToken *tptr, const PGPByte *keyid, PGPBoolean is_last )
{
	return 0;
}

static int
ibWipe( PGPToken *tptr )
{
  LPRESPONSEAPDU  apdu;
  BYTE			  cmdbuf[1000];

  apdu = Process(PGPKEY_CLA, PGPKEY_INS_INIT,
	  0, 0, 0, NULL, LOAD_UNLOAD_TIME);
  
  if(apdu->SW != (WORD)0x9000)
  {
    return -1;
  }

  cmdbuf[0] = 0; /* old pin */
  memcpy( cmdbuf+1, tokenPin, tokenPinLen );
  apdu = Process(PGPKEY_CLA, PGPKEY_INS_SETPIN,
	  0, 0, (BYTE)(tokenPinLen+1), cmdbuf, LOAD_UNLOAD_TIME);

  if(apdu->SW != (WORD)0x9000)
  {
    return -1;
  }

  nkeys = 0;
  memset( tokenKeyidBlob, 0, KEYIDSIZE );

  return 0;
}

static int
ibChangeKeyID( PGPToken *tptr, PGPByte *oldkeyid,
			  PGPByte *newkeyid )
{
  LPRESPONSEAPDU  apdu;
  BYTE			  cmdbuf[1000];
  int			  keyindex;

  keyindex = ibKeyIDIndex( oldkeyid );
  if (keyindex < 0)
	  return -1;

  cmdbuf[0] = tokenPinLen;
  memcpy( cmdbuf+1, tokenPin, tokenPinLen );
  cmdbuf[tokenPinLen+1] = keyindex;
  memcpy(cmdbuf+tokenPinLen+2, newkeyid, KEYIDSIZE);
  apdu = Process(PGPKEY_CLA, PGPKEY_INS_SETKEYID,
	  0, 0, (BYTE)(tokenPinLen+2+8), cmdbuf, LOAD_UNLOAD_TIME);

  if(apdu->SW != (WORD)0x9000)
  {
    printf("SetKeyID failed with SW = %04x\n",apdu->SW);
    return FALSE;
  }

  memcpy( tokenKeyids[keyindex], newkeyid, KEYIDSIZE );
  memcpy( keyinfo[keyindex].keyid, newkeyid, KEYIDSIZE );

  if (memcmp( tokenKeyidBlob, oldkeyid, KEYIDSIZE ) == 0) {
	  /* Also update blob keyid */
	cmdbuf[tokenPinLen+1] = 0xff;
	apdu = Process(PGPKEY_CLA, PGPKEY_INS_SETKEYID,
		0, 0, (BYTE)(tokenPinLen+2+8), cmdbuf, LOAD_UNLOAD_TIME);

	if(apdu->SW != (WORD)0x9000)
	{
		printf("SetKeyID failed with SW = %04x\n",apdu->SW);
		return FALSE;
	}
  }
  return 0;
}

static int
ibSetPIN( PGPToken *tptr, char *oldpin, PGPSize oldpinsize,
			  char *newpin, PGPSize newpinsize )
{
	LPRESPONSEAPDU  apdu = NULL;
	BYTE bigbuf[1000];

	if( (int)oldpinsize != tokenPinLen ||
		memcmp( oldpin, tokenPin, oldpinsize ) != 0 )
		return -1;

	bigbuf[0] = (BYTE) oldpinsize;
	memcpy( bigbuf+1, oldpin, oldpinsize );
	memcpy( bigbuf+1+oldpinsize, newpin, newpinsize );

	apdu = Process(PGPKEY_CLA, PGPKEY_INS_SETPIN,
			0, 0, (BYTE)(1+oldpinsize+newpinsize), bigbuf, LOAD_UNLOAD_TIME);

	if(apdu->SW != (WORD)0x9000)
		return 1;
	memcpy( tokenPin, newpin, newpinsize );
	tokenPinLen = newpinsize;
	tokenPinChecked = 1;
	return 0;
}

static int
ibKeygen( PGPToken *tptr, PGPByte *keyid, PGPSize modulusSize,
		 PGPBoolean genMaster, PGPByte *modulus, PGPSize *pubexp)
{
	LPRESPONSEAPDU  apdu = NULL;
	BYTE bigbuf[1000];
	int keyindex;

  if (nkeys >= NKEYS)
	  return -1;

	keyindex = nkeys;

	bigbuf[0] = tokenPinLen;
	memcpy( bigbuf+1, tokenPin, tokenPinLen );
	bigbuf[tokenPinLen+1] = (PGPByte)keyindex;
	bigbuf[tokenPinLen+2] = (PGPByte)((modulusSize>>8) & 0xff);
	bigbuf[tokenPinLen+3] = (PGPByte)(modulusSize & 0xff);

	apdu = Process(PGPKEY_CLA, PGPKEY_INS_RSAKEYGEN,
		0, 0, (BYTE)(tokenPinLen+4), bigbuf, KEYGEN_TIME);

	if(apdu->SW != (WORD)0x9000)
		return -1;

	if (apdu->Len > modulusSize/8 )
		return -1;
	if( apdu->Len < modulusSize/8 )
		memset( modulus, 0, modulusSize/8 - apdu->Len );
	memcpy( modulus + modulusSize/8 - apdu->Len,
				apdu->Data, apdu->Len );
	*pubexp = 0x10001;

	if( genMaster )
	{
		/* Store public exponent in public blob */
		memcpy( bigbuf, keyid, KEYIDSIZE );
		bigbuf[KEYIDSIZE+0] = 0x00;
		bigbuf[KEYIDSIZE+1] = 0x03;
		bigbuf[KEYIDSIZE+2] = 0x01;
		bigbuf[KEYIDSIZE+3] = 0x00;
		bigbuf[KEYIDSIZE+4] = 0x01;
		apdu = Process(PGPKEY_CLA, PGPKEY_INS_SETBLOB,
			0, 0, KEYIDSIZE+2+3, bigbuf, LOAD_UNLOAD_TIME);
		if(apdu->SW != (WORD)0x9000)
		{
			printf("SetBlob failed with SW = %04x\n",apdu->SW);
			return -3;
		}
		memcpy( tokenKeyidBlob, keyid, KEYIDSIZE );
	}

	memcpy( tokenKeyids[keyindex], keyid, KEYIDSIZE );
	memcpy( keyinfo[keyindex].keyid, keyid, KEYIDSIZE );
	++nkeys;

	return 0;
}

static PGPError
ibGetInfo( PGPToken *tptr, PGPTokenInfo *info )
{
	*info = ourToken.info;
	return 0;
}

static PGPTokenCertInfo *
ibGetX509Certs( PGPToken *tptr, PGPTokenKeyInfo *info, PGPInt32 *n )
{
	*n = 0;
	return malloc(0);
}


static struct PGPToken ourToken = {
	ibInit,
	ibTerm,
	ibDeauth,
	ibAuth,
	ibDecrypt,
	ibSign,
	ibGetPrivKeyIDs,
	ibGetPubKeyIDs,
	ibGetPublic,
	ibGetX509Certs,
	NULL,
	ibPutPublic,
	ibPutPrivate,
	ibDel,
	ibKeygen,
	ibChangeKeyID,
	ibSetPIN,
	ibWipe,
	ibGetInfo,
	NULL,
	NULL, /* putkeycontainer */
	NULL, /* getkeycontainer */
	FALSE, 0, FALSE, 0, FALSE,
	{sizeof(PGPTokenInfo), "iButton", "", "",
	TRUE, 512, 1024, 0, 128},
};


/* EXTERNAL ENTRY POINTS */


/* This returns TRUE if there is a change in the set of plugged-in tokens */
int
pgpAcquireAllTokens(void)
{
	LPRESPONSEAPDU  apdu = NULL;
	/* number of JiBs found */
	BYTE    jibCount;

	SetJiBSearchParams(g_SerialPortCount, g_DeviceNames); 

	/* Search adapters for attached JiBs */
	roms = FindJiBs(&jibCount);

	if (gottoken && jibCount == 0) {
		nkeys = 0;
		gottoken = 0;
		return 1;
	} else if (!gottoken && jibCount > 0) {
		byte cmdbuf[1];
		int i;

		SelectJiB(roms);
		for( i=0; i<NKEYS; ++i )
		{
			cmdbuf[0] = (PGPByte)i;
			apdu = Process(PGPKEY_CLA, PGPKEY_INS_GETKEYID,
				0, 0, 1, cmdbuf, LOAD_UNLOAD_TIME);
  
			if(apdu->SW != (WORD)0x9000)
				break;
  
			if( apdu->Len != KEYIDSIZE )
				break;
			memcpy( tokenKeyids[i], apdu->Data, KEYIDSIZE );
			memcpy( keyinfo[i].keyid, apdu->Data, KEYIDSIZE );
		}
		nkeys = i;

		cmdbuf[0] = (PGPByte)0xff;
		apdu = Process(PGPKEY_CLA, PGPKEY_INS_GETKEYID,
			0, 0, 1, cmdbuf, LOAD_UNLOAD_TIME);

		memset( tokenKeyidBlob, 0, KEYIDSIZE );
		pubkeyinfo.pgpData = TRUE;
		if(apdu->SW == (WORD)0x9000
			&& apdu->Len == KEYIDSIZE )
			memcpy( tokenKeyidBlob, apdu->Data, KEYIDSIZE );

		/* Turn on garbage collection on token */
		apdu = SetAppletGCMode(1);

		tokenPinChecked = 0;
		tokenPinLen = 0;
		gottoken = 1;
		return 1;
	}
	return 0;
}

void
pgpReleaseAllTokens(void)
{
	nkeys = 0;
	gottoken = 0;
}

PGPError
pgpTokenInit(void)
{
	nkeys = 0;
	gottoken = 0;
	return 0;
}

struct PGPToken *
pgpGetTokenObjectByPrivKeyID(const BYTE *keyid)
{
	if( nkeys == 0 )
		return NULL;

	if( ibKeyIDIndex( keyid ) < 0 )
		return NULL;
  
  return &ourToken;
}

/*
struct PGPToken *
pgpGetTokenObjectByPubKeyID(const BYTE *keyid)
{
	if( nkeys == 0 )
		return NULL;
	if( memcmp( tokenKeyidBlob, keyid, KEYIDSIZE ) != 0 )
		return NULL;

	return &ourToken;
}
*/

PGPUInt32
pgpCountTokenObjects(void)
{
	return gottoken ? 1 : 0;
}

struct PGPToken *
pgpGetNthTokenObject(PGPUInt32 n)
{
	if( !gottoken || n > 0 )
		return 0;
	return &ourToken;
}

PGPError pgpFreeMem( void *p )  {
    free(p);
    return 0;
}


